/**
  ******************************************************************************
  * @file    py32f410_ll_opa.h
  * @author  MCU Application Team
  * @brief   Header file of OPA LL module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __PY32F410_LL_OPA_H
#define __PY32F410_LL_OPA_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32f4xx.h"

/** @addtogroup PY32F410_LL_Driver
  * @{
  */

#if defined (OPA1) || defined (OPA2) 

/** @defgroup OPA_LL OPA
  * @{
  */

/* Private types -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

/* Private constants ---------------------------------------------------------*/
/** @defgroup OPA_LL_Private_Constants OPA Private Constants
  * @{
  */

/** @defgroup OPA_LL_EC_OUTPUT_CONNECTION  OPA output connection
  * @{
  */
#define LL_OPA_OUTPUT_CONNECT_GPIO         0x00000000UL            /*!< The selected OPA output is connected to external pin */
#define LL_OPA_OUTPUT_CONNECT_INTERNAL     (OPA_CSR_OPAINTOEN)     /*!< The selected OPA output is connected to on-chip peripherals via internal paths */
/**
  * @}
  */

/** @defgroup OPA_LL_EC_MODE OPA Mode
  * @{
  */
#define LL_OPA_MODE_OPA            (0x00000000UL)                /*!< Mode OPA */
#define LL_OPA_MODE_COMP           (OPA_CSR_OPA_CMP_CR)          /*!< Mode COMP */
/**
  * @}
  */

/** @defgroup OPA_LL_EC_OUTPUT_LEVEL OPA output - Output level
  * @{
  */
#define LL_OPA_OUTPUT_LEVEL_LOW            (0x00000000U)          /*!< OPA Comparator output level low  */
#define LL_OPA_OUTPUT_LEVEL_HIGH           (0x00000001U)          /*!< OPA Comparator output level high */

/**
  * @}
  */

/**
  * @}
  */

/* Private macros ------------------------------------------------------------*/
/** @defgroup OPA_LL_Private_Macros OPA Private Macros
  * @{
  */
  
/**
  * @}
  */


/* Exported types ------------------------------------------------------------*/
#if defined(USE_FULL_LL_DRIVER)


#endif /* USE_FULL_LL_DRIVER */

/* Exported constants --------------------------------------------------------*/

/* Exported functions --------------------------------------------------------*/
/** @defgroup OPA_LL_Exported_Functions OPA Exported Functions
  * @{
  */

/** @defgroup OPA_LL_EF_CONFIGURATION Configuration
  * @{
  */

/**
  * @brief  Enable the OPA.
  * @rmtoll CSR      OPAEN              LL_OPA_Enable
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_Enable(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->CSR, OPA_CSR_OPAEN);
}

/**
  * @brief  Disable the OPA.
  * @rmtoll CSR      OPAEN              LL_OPA_Disable
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_Disable(OPA_TypeDef *OPAx)
{
  CLEAR_BIT(OPAx->CSR, OPA_CSR_OPAEN);
}

/**
  * @brief  Get the OPA1 Enable state.
  * @rmtoll CSR      OPAEN              LL_OPA_IsEnabled
  * @param  OPAx OPA instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_OPA_IsEnabled(OPA_TypeDef *OPAx)
{
  return (READ_BIT(OPAx->CSR, OPA_CSR_OPAEN) == OPA_CSR_OPAEN);
}

/**
  * @brief  Set the output connection for the selected OPA.
  * @rmtoll CSR       OPAINTOEN          LL_OPA_SetOutputConnection
  * @param  OPAx OPA instance
  * @param  OutputConnection This parameter can be one of the following values:
  *         @arg @ref LL_OPA_OUTPUT_CONNECT_GPIO
  *         @arg @ref LL_OPA_OUTPUT_CONNECT_INTERNAL
  * @retval None
  */
__STATIC_INLINE void LL_OPA_SetOutputConnection(OPA_TypeDef *OPAx,uint32_t OutputConnection)
{
  MODIFY_REG(OPAx->CSR,OPA_CSR_OPAINTOEN,OutputConnection);
}


/**
  * @brief  Get the output connection for the selected OPA.
  * @rmtoll CSR       OPAINTOEN          LL_OPA_GetOutputConnection
  * @param  OPAx OPA instance
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_OPA_OUTPUT_CONNECT_GPIO
  *         @arg @ref LL_OPA_OUTPUT_CONNECT_INTERNAL
  */
__STATIC_INLINE uint32_t LL_OPA_GetOutputConnection(OPA_TypeDef *OPAx)
{
  return (uint32_t)(READ_BIT(OPAx->CSR,OPA_CSR_OPAINTOEN));
}

/**
  * @brief  Read comparator instance output level.
  * @rmtoll CSR      OPA_COMP_OUT       LL_COMP_ReadOutputLevel
  * @param  OPAx Comparator instance
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_OPA_OUTPUT_LEVEL_LOW
  *         @arg @ref LL_OPA_OUTPUT_LEVEL_HIGH   
  */
__STATIC_INLINE uint32_t LL_OPA_ReadOutputLevel(OPA_TypeDef *OPAx)
{
  return (uint32_t)(READ_BIT(OPAx->CSR, OPA_CSR_OPA_CMP_OUT) >> OPA_CSR_OPA_CMP_OUT_Pos);
}

/**
  * @}
  */

/**
  * @brief  Set opa mode.
  * @rmtoll CSR       FLTCNT         LL_OPA_SetMode
  * @param  OPAx OPA instance      
  * @param  OPA_Mode This parameter can be one of the following values:
  *         @arg @ref LL_OPA_MODE_OPA
  *         @arg @ref LL_OPA_MODE_COMP
  * @retval None
  */
__STATIC_INLINE void LL_OPA_SetMode(OPA_TypeDef *OPAx,uint32_t OPA_Mode)
{
  MODIFY_REG(OPAx->CSR,OPA_CSR_OPA_CMP_CR,OPA_Mode);
}

/**
  * @brief  Get opa mode
  * @rmtoll CSR       OPA_CMP_CR         LL_OPA_GetMode
  * @param  OPAx OPA instance          
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_OPA_MODE_OPA
  *         @arg @ref LL_OPA_MODE_COMP   
  */
__STATIC_INLINE uint32_t LL_OPA_GetMode(OPA_TypeDef *OPAx)
{
    return (uint32_t)(READ_BIT(OPAx->CSR,OPA_CSR_OPA_CMP_CR));
}

/**
  * @brief  Enable interruption OPA COMP Rising Trigger.
  * @rmtoll INTR      INTEN          LL_OPA_EnableIT_RisingTrig
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_EnableIT_RisingTrig(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->INTR, OPA_INTR_INTEN);
}

/**
  * @brief  Disable interruption OPA COMP Rising Trigger.
  * @rmtoll INTR      INTEN          LL_OPA_DisableIT_RisingTrig
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_DisableIT_RisingTrig(OPA_TypeDef *OPAx)
{
  CLEAR_BIT(OPAx->INTR, OPA_INTR_INTEN);
}

/**
  * @brief  Get state of interruption OPA COMP Rising Trigger.
  *         (0: interrupt disabled, 1: interrupt enabled).
  * @rmtoll INTR      INTEN          LL_OPA_IsEnabledIT_RisingTrig
  * @param  OPAx OPA instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_OPA_IsEnabledIT_RisingTrig(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->INTR, OPA_INTR_INTEN) == (OPA_INTR_INTEN)) ? 1UL : 0UL);
}

/**
  * @brief  Get flag OPA COMP Rising Trigger.
  * @rmtoll INTR      INTIF           LL_OPA_IsActiveFlag_RisingTrig
  * @param  OPAx OPA instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_OPA_IsActiveFlag_RisingTrig(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->INTR, OPA_INTR_INTIF) == (OPA_INTR_INTIF)) ? 1UL : 0UL);
}

/**
  * @brief  Clear flag OPA COMP Rising Trigger.
  * @rmtoll INTR      CINTIF          LL_OPA_ClearFlag_RisingTrig
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_ClearFlag_RisingTrig(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->INTR, OPA_INTR_CINTIF);
}

/**
  * @brief  Lock the OPA.
  * @rmtoll CSR      LOCK          LL_OPA_Lock
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_Lock(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->CSR, OPA_CSR_LOCK);
}

/**
  * @brief  Get flag OPA Locked.
  * @rmtoll CSR      LOCK           LL_OPA_IsLocked
  * @param  OPAx ADC instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_OPA_IsLocked(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->CSR, OPA_CSR_LOCK) == (OPA_CSR_LOCK)) ? 1UL : 0UL);
}

/** @defgroup OPA_LL_EF_DigitalFilter DigitalFilter on opa instance
  * @{
  */
/**
  * @brief  Enable OPA DigitalFilter.
  * @rmtoll FR       FLTEN          LL_COMP_EnableDigitalFilter
  * @param  OPAx OPA instance   
  * @retval None
  */
__STATIC_INLINE void LL_OPA_EnableDigitalFilter(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->FR, OPA_FR_FLTEN);
}

/**
  * @brief  Disable OPA DigitalFilter.
  * @rmtoll FR       FLTEN          LL_OPA_DisableDigitalFilter
  * @param  OPAx OPA instance   
  * @retval None
  */
__STATIC_INLINE void LL_OPA_DisableDigitalFilter(OPA_TypeDef *OPAx)
{
  CLEAR_BIT(OPAx->FR, OPA_FR_FLTEN);
}

/**
  * @brief  Get OPA DigitalFilter state
  *         (0: Filter is Disabled, 1: Filter is Enabled).
  * @rmtoll FR       FLTEN          LL_OPA_IsEnabledDigitalFilter
  * @param  OPAx OPA instance    
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_OPA_IsEnabledDigitalFilter(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->FR, OPA_FR_FLTEN) == (OPA_FR_FLTEN)) ? 1U : 0U);
}

/**
  * @brief  Set OPA DigitalFilter Value.
  * @rmtoll FR       FLTCNT         LL_OPA_SetDigitalFilter
  * @param  OPAx OPA instance      
  * @param  FLTCNTValue Value between Min_Data=0x0000 and Max_Data=0xFFFF
  * @retval None
  */
__STATIC_INLINE void LL_OPA_SetDigitalFilter(OPA_TypeDef *OPAx,uint32_t FLTCNTValue)
{
  MODIFY_REG(OPAx->FR,OPA_FR_FLTCNT,FLTCNTValue << OPA_FR_FLTCNT_Pos);
}

/**
  * @brief  Get OPA DigitalFilter Value
  * @rmtoll FR       FLTCNT         LL_OPA_GetDigitalFilter
  * @param  OPAx OPA instance          
  * @retval DigitalFilter Value between Min_Data=0x0000 and Max_Data=0xFFFF
  */
__STATIC_INLINE uint32_t LL_OPA_GetDigitalFilter(OPA_TypeDef *OPAx)
{
  return (uint32_t)(READ_REG(OPAx->FR)>>OPA_FR_FLTCNT_Pos);
}

/**
  * @}
  */

/**
  * @}
  */

#if defined(USE_FULL_LL_DRIVER)
/** @defgroup OPA_LL_EF_Init Initialization and de-initialization functions
  * @{
  */

/**
  * @}
  */
#endif /* USE_FULL_LL_DRIVER */

/**
  * @}
  */

/**
  * @}
  */

#endif /* OPA */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __PY32F410_LL_OPA_H */

/************************ (C) COPYRIGHT Puya *****END OF FILE****/
